package MyLibrary::Auth;

use MyLibrary::Patron;
use MyLibrary::Config;
use MyLibrary::DB;
use CGI::Session qw/-ip-match/;
use Carp qw(croak);

use strict;

=head1 NAME

MyLibrary::Auth

=head1 SYNOPSIS

        use MyLibrary::Auth;

=head1 DESCRIPTION

Authentication system for MyLibrary.

=cut


# Stores references to hashes containing object data
my %_auth_obj;

	{
		# Allowable object attributes with defaults
		my %_attr_data =
			(	sessid			=> undef,
			  	status			=> 'not authenticated',
				user_id			=> undef,
				username		=> undef,
				session_expire	=> undef,
				file			=> __FILE__ 
			);
		# Class methods used to operate on encapsulated data
		sub _attr_defaults {
			my $sessid = shift;
			$_attr_data{'sessid'} = $sessid;
			return \%_attr_data;
		}

		sub _standard_keys {
			keys %_attr_data;
		}
	}


sub new {
	my ($self, %args) = @_;
	my $class = ref($self) || $self;
	if (my $sessid = $args{sessid}) {
		my $session = CGI::Session->new("driver:File", $sessid, {Directory=>$MyLibrary::Config::SESSION_DIR});
		my $_attr_flds_ref = {};
		my $_session_params = $session->param_hashref();
		foreach my $attr (keys %$_session_params) {
			$_attr_flds_ref->{$attr} = %$_session_params->{$attr};
		}
		$_attr_flds_ref->{status_accessor} = sub {
			my $self = shift;
			my $status = shift;
			if (defined($status) && $status =~ /^not authenticated$|^authenticated$|^failed authentication$|^expired$/) {
				return $_auth_obj{${$self}}->{status} = $status;
			} else {
				return $_auth_obj{${$self}}->{status};
			}
		};
		$_attr_flds_ref->{_sess_ref} = $session;
		$_attr_flds_ref->{_key} = rand
			until $_attr_flds_ref->{_key} && !exists $_auth_obj{$_attr_flds_ref->{_key}};
		$_auth_obj{$_attr_flds_ref->{_key}} = $_attr_flds_ref;
		return bless(\$_attr_flds_ref->{_key}, $class);		
	} else {
		my $session = CGI::Session->new("driver:File", undef, {Directory=>$MyLibrary::Config::SESSION_DIR});
		my $sessid = $session->id();
		my $_base_attr_fields = _attr_defaults($sessid);
		my $_attr_fields = $self->_attr_defaults();
		my $_attr_flds_ref = {%$_base_attr_fields, %$_attr_fields};
		foreach my $attr (keys %$_attr_flds_ref) {
			$_attr_flds_ref->{$attr} = $args{$attr} if defined $args{$attr};
			$session->param($attr, $_attr_flds_ref->{$attr});
		}
		$_attr_flds_ref->{status_accessor} = sub {
			my $self = shift;
			my $status = shift;
			if (defined($status) && $status =~ /^not authenticated$|^authenticated$|^failed authentication$|^expired$/) {
				return $_auth_obj{${$self}}->{status} = $status;
			} else {
				return $_auth_obj{${$self}}->{status};
			}
		};
		$_attr_flds_ref->{_sess_ref} = $session;
		$_attr_flds_ref->{_key} = rand
			until $_attr_flds_ref->{_key} && !exists $_auth_obj{$_attr_flds_ref->{_key}};
		$_auth_obj{$_attr_flds_ref->{_key}} = $_attr_flds_ref;
		return bless(\$_attr_flds_ref->{_key}, $class);
	}
}

sub sessid {
	my $self = shift;
	return $_auth_obj{${$self}}->{sessid};
}

sub status {
	my $self = shift;
	if ($_auth_obj{${$self}}->{status_accessor}->($self) eq 'authenticated') {
		unless ($self->_logged_in()) {
			$_auth_obj{${$self}}->{status_accessor}->($self, 'expired');
			return $_auth_obj{${$self}}->{status_accessor}->($self);
		}
		return $_auth_obj{${$self}}->{status_accessor}->($self);
	} else {
		return $_auth_obj{${$self}}->{status_accessor}->($self);
	}
}

sub user_id {
	my $self = shift;
	return $_auth_obj{${$self}}->{user_id};
}

sub username {
	my $self = shift;
	return $_auth_obj{${$self}}->{username};
}

sub _logged_in {
	my $self = shift;
	return $_auth_obj{${$self}}->{_sess_ref}->param('_logged_in');
}

sub place_cookie {
	my $self = shift;
	print $self->_header();	
}

sub remove_cookie {
	my $self = shift;
	print $self->_header(action => 'remove');
}

sub _header {
	my $self = shift;
	my %args = @_;
	my $session = $_auth_obj{${$self}}->{_sess_ref};
	my $expire_time;
	my $cgi = $session->{_SESSION_OBJ};
	unless ( defined $cgi ) {
		require CGI;
		$session->{_SESSION_OBJ} = CGI->new();
		$cgi = $session->{_SESSION_OBJ};
	}
	if (defined $args{action} && $args{action} eq 'remove') {
		$expire_time = '-1d';
	} else {
		$expire_time = '+10M';
	}
	my $cookie = $cgi->cookie(-name=>'mylib_sessid',-value=>$session->id(), -path=>$MyLibrary::Config::RELATIVE_PATH,
								-domain=>$MyLibrary::Config::COOKIE_DOMAIN, -expires=>$expire_time);
	return $cgi->header(
		-type   => 'text/html',
		-cookie => $cookie,
		@_
	);
}

sub _attr_hash {
	my $self = shift;
	my @caller = caller();
	if ($caller[0] eq 'main' || $caller[0] !~ /^MyLibrary::Auth::\w+/ || $caller[1] ne $_auth_obj{${$self}}->{file}) {
		croak "Illegal call to private MyLibrary::Auth method";
	}
	return \%_auth_obj;
}

1;
