#!/usr/bin/perl

# portal.cgi - an interface to a list of Internet resources

# Eric Lease Morgan <emorgan@nd.edu>
# January 27, 2006 - Moving to MyLibrary distro
# September 18, 2005 - First cut


=head1 NAME

portal.cgi - an example MyLibrary application


=head1 DESCRIPTION

This script (portal.cgi) is the root of a system of files allowing an end-user to access and use a collection of Internet resources. The script supports simple browse and search functionalities. The script requires very little configuration. In order to customize the system you will need to become familiar with the templates and tokens it supports. See below.

The creation and maintenance of the collection is not supported by this script but by another set of files found the cgi-bin directory. 


=head2 Configuration

Configuration requires the redefinition of SRUROOT in the head of the script. Change this value so the URL points to the location of your SRU-accessible index. You will want to change the host in the URL to your host, and you will want to change the initial path to the SRU server to reflect the location of server.cgi in your HTTP hierarchy.

The other constants need not be edited. Most of them simply point the location of template files.


=head2 Templates and tokens

Customizing the user interface is mostly a matter of editing a number of template files and making sure they contain specific tokens. Template files are snippets of HTML. Tokens are case-sensitive placeholders in the templates that get replaced with content. Each token is surrounded by a pair of hash marks (##) in order to designate it as a token. Each template file and its associated set of tokens is described below:


=over 4

=item * etc/templagte.txt

This the root template. It defines the whole look & feel of the system. It is expected to include your HTML declaration, your header, and your footer. It is in this template where you define the location of your CSS files. It is in this file where your graphic designer will do their most work.

This template includes only two tokens: 1) CONTENT, and 2) VERSION. CONTENT gets replaced with the HTML of all the other templates. In other words, wherever you put the CONTENT token is where the content of the home page will appear, where the content of the about page will appear, where the content of the search results will appear, etc. VERSION gets replaced with the version number of your MyLibrary Perl modules. Your users won't really care about this, so you might decide to not include it at all.


=item * etc/home.txt

This template is the system's home page; it is the text that gets displayed by default. It is where you are expected to briefly describe the scope of your collection and what the user can expect to find here.

To get the user started you might want to display a hot-linked list of your facets and terms, and the tokens enable you to do just that. The first token, NUMBER_OF_RESOURCES, simply returns an integer denoting the number if resources in your MyLibrary implementation. The second token, FACETS, returns a hierarchal list of your facet/term combinations, short descriptions of each facet/term, as well as the number of resources associated with each facet. In other words, FACETS gets replaced with a simple browsable list of resources.


=item * etc/about.txt

This is intended to be your About page. Insert into this template a more elaborate description of your collection and who end-users can call for help. This template includes no tokens.


=item * etc/term.txt

This template is the result of a user selecting a hot-linked term value from the browsable list of facet/term combinations. It contains three tokens: 1) TERM, 2) TERM_NOTE, and 3) RESOURCE_LIST.

TERM is the name of the term in the facet/term combination. TERM_NOTE is the definition of the TERM as denoted in the administrative interface. RESOURCE_LIST is an alphabetical (ordered) list of information resources associated with the term. Each item in the list will include the resource's hot-linked title and description.


=item * etc/search.txt

This is the search results page. Once a query is sent to the system this page will include the list of resources matching the query. This page has only one token, RESULTS, and this is where the results will be displayed.

To change the format of the list of resources you will need to edit the XSL stylesheet (etc/sru2html.xsl). Describing how to do this is beyond the scope of this documentation.


=back


=head2 Stylesheets

Additional customizations of the end-user interface can be gotten by editing the stylesheets. There are two types. The cascading stylesheets (etc/portal-screen.css and etc/portal-print.css) are used to change the look & feel of the whole site. The XSL stylesheet (etc/sru2html.xsl) is used to transform the SRU search results into an HTML ordered list. Alas, describing how to edit these files is beyond the scope of this documentation.


=head2 Commands

The script is driven by three commands sent as HTTP GET or POST requests. Each command is in the form a name/value pair where the first part of the pair is 'cmd' and the second part is the command itself.

=over 4

=item 1) about

Use this command to hyperlink to your About page. For example, consider putting something like this into etc/template.txt file:

  Read <a href='./?cmd=about'>about</a> this system.


=item 2) search

Use this command to initiate a query against MyLibrary. The command requires an additional name/value pair denoting the query. A simple search form might look like this:

  <form method='post' action='./'>
    <input type='hidden' name='cmd' value='search' />
    <input name='query' size='15' /><br />
    <input type='submit' value='Go' />
  </form>

A "canned" search might look like this:

  Search for <a href='./?cmd=search&query=dogs'>dogs</a>.


=item 3) term

The term command is used to display information resources associated with a term. This command requires an additional name/value pair denoting the ID of the term to display. Since the term command is embedded in the browsable display of facet/term combinations, there will be little need for you to put this command into your templates, but if you did, then it might look something like this:

  List resources associated with <a href='./cmd=term&id=23'>Astronomy</a>.
  
  
=back


=head1 AUTHOR

Eric Lease Morgan

=cut

# define constants
use constant ABOUT        => 'etc/about.txt';
use constant CMDTERM      => 'term';
use constant HOME         => 'etc/home.txt';
use constant LOGIN        => 'etc/login.txt';
use constant PATRON       => 'etc/patron.txt';
use constant SEARCH       => 'etc/search.txt';
use constant SRU2HTML     => 'etc/sru2html.xsl';
use constant SRUROOT      => 'http://dewey.library.nd.edu/morgan/zagreb/sru/server.cgi?operation=searchRetrieve&version=1.1&query=';
use constant TEMPLATE     => 'etc/template.txt';
use constant TERM         => 'etc/term.txt';
use constant STYLESHEETID => 1;
use constant SUBJECTS     => 5;


# require/use
use lib './lib';
use CGI;
use CGI::Carp qw(fatalsToBrowser);
use MyLibrary::Core;
use MyLibrary::Portal;
use strict;
use LWP::UserAgent;
use XML::LibXML;
use XML::LibXSLT;

# a hack to see who is using this application
#open (MAIL, "| mail -s '[patron_test] $ENV{REMOTE_HOST}' emorgan\@nd.edu");
#print MAIL qq($ENV{REMOTE_ADDR} ($ENV{REMOTE_HOST})\n$ENV{HTTP_USER_AGENT}\n\n$ENV{QUERY_STRING});
#close MAIL;

# initialize
my $cgi = CGI->new;
my $html;

# get the command
my $cmd = $cgi->param('cmd');

# branch accordingly
if (! $cmd) {

	# display the home page
	$html = &slurp(TEMPLATE);
	$html =~ s/##CONTENT##/&slurp(HOME)/e;
	$html =~ s/##NUMBER_OF_RESOURCES##/&number_of_items/e;
	$html =~ s/##FACETS##/&facet_term_combinations/e;
		
}

elsif ($cmd eq 'about') {

	# display the about page
	$html = &slurp(TEMPLATE);
	$html =~ s/##CONTENT##/&slurp(ABOUT)/e;
		
}

elsif ($cmd eq 'create') {

	# create an object, fill it, and save
	my $patron = MyLibrary::Patron->new;
	$patron->patron_firstname($cgi->param('firstname'));
	$patron->patron_surname($cgi->param('surname'));
	$patron->patron_email($cgi->param('email'));
	$patron->patron_username($cgi->param('username'));
	$patron->patron_password($cgi->param('password'));
	$patron->patron_stylesheet_id(STYLESHEETID);
	$patron->commit();
	
	# get the terms and suggested resources for this patron
	my @term_ids;
	my $terms;
	my @suggested_resources;
	foreach ($cgi->param('term_names')) {
	
		# get the selected term
		my $term = MyLibrary::Term->new(name => $_);
		
		# save the id and name for future reference
		push @term_ids, $term->term_id;
		$terms .= $term->term_name . '; ';
		
		# get the suggested resources for this term
		push @suggested_resources, $term->suggested_resources;
		
	}
	
	# "catalog" the patron
	$patron->patron_terms(new => [@term_ids]);
	
	# give them a set of recommended resources
	@suggested_resources = &MyLibrary::Portal::remove_duplicates(@suggested_resources);
	$patron->patron_resources(new => [@suggested_resources]);

	# display the home page
	$html = &slurp(TEMPLATE);
	$html =~ s/##CONTENT##/&slurp(HOME)/e;
	$html =~ s/##NUMBER_OF_RESOURCES##/&number_of_items/e;
	$html =~ s/##FACETS##/&facet_term_combinations/e;

}

elsif ($cmd eq 'login') {

	# build a list of subject terms
	my $facet = MyLibrary::Facet->new( id => SUBJECTS );
	my @related_term_ids = $facet->related_terms(sort => 'name');
	my $term_names;
	foreach my $related_term_id (@related_term_ids) {
	
		# create the term and build the option list
		my $term = MyLibrary::Term->new(id => $related_term_id);
		$term_names .= '<option value="' . $term->term_name . '">' . $term->term_name . '</option>';
	
	}
	
	# display the login page
	$html = &slurp(TEMPLATE);
	$html =~ s/##CONTENT##/&slurp(LOGIN)/e;
	$html =~ s/##TERMNAMES##/$term_names/e;
		
}

elsif ($cmd eq 'term') {

	# geth the input and find the term
	my $term_id = $cgi->param('id');
	my $term = MyLibrary::Term->new( id => $term_id );

	# build an html list of resources for the input
	my @resource_ids = $term->related_resources( sort => 'name' );
	my $list;
	foreach my $id ( @resource_ids ) {
	
		# get this resouce
		my $resource = MyLibrary::Resource->new( id => $id );
		
		# get the url; dumb!
		my @locations = $resource->resource_locations;
		my $location;
		foreach ( @locations ) { $location = $_->location }
		
		# build a list
		$list .= $cgi->li($cgi->a({-href => $location}, $resource->name) . ' - ' . $resource->note);
		
	}
	$list = $cgi->ol($list);
	
	# display the term page
	$html = &slurp(TEMPLATE);
	$html =~ s/##CONTENT##/&slurp(TERM)/e;
	$html =~ s/##TERM##/$term->term_name/e;
	$html =~ s/##TERM_NOTE##/$term->term_note/e;
	$html =~ s/##RESOURCE_LIST##/$list/e;
	
}
		
elsif ($cmd eq 'search') {
	
	# get the input
	my $query = $cgi->param('query');
		
	# begin to complete the search results page
	$html = &slurp(TEMPLATE);
	$html =~ s/##CONTENT##/&slurp(SEARCH)/e;
			
	# no query supplied; therefore, no results
	if ( ! $query ) { $html =~ s/##RESULTS##// }
	
	# process the query
	else {
	
		# do simple munging of the query; try to force it into CQL
		if ($query =~ /\s/) {
		
			if (($query =~ / and /) | ($query =~ / or /) | ($query =~ / not /)) { }		
			elsif ($query =~ /=/) { }
			elsif ($query !~ /"/) {
			
				# try to make queries with no syntactical sugar a bit "smarter"
				my @terms = split / /, $query;
				my $enhancement;
				for (my $i; $i <= $#terms; $i++) {
				
					if ($i < $#terms) { $enhancement .= $terms[$i] . ' and ' }
					else { $enhancement .= $terms[$i] }
				
				}
				
				$query = '"' . $query . '"' . " or ($enhancement)";
				
			}
			
		}
		
		# create a user agent, create a request, send it, and get a response
		my $ua = LWP::UserAgent->new(agent => 'SRU-Client/0.1 ');
		my $request = HTTP::Request->new(GET => SRUROOT . $query);
		my $response = $ua->request($request);
		
		# check response
		if ($response->is_success) {
		
			# create parser and process input
			my $parser     = XML::LibXML->new;
			my $xslt       = XML::LibXSLT->new;
			my $source     = $parser->parse_string($response->content) or croak $!;
			my $style      = $parser->parse_file(SRU2HTML)             or croak $!;
			my $stylesheet = $xslt->parse_stylesheet($style)           or croak $!;
			my $results    = $stylesheet->transform($source)           or croak $!;
			
			# update the output
			$html =~ s/##RESULTS##/$stylesheet->output_string($results)/e;
						
		}
		
		else { $html = $response->status_line, "\n" }
	
	}
	
}


elsif ($cmd eq 'patron') {

	# create a patron based on the id
	my $patron = MyLibrary::Patron->new(id => $cgi->param('id'));
	
	# create a resource list
	my $resource_list;
	my @resource_ids = $patron->patron_resources(sort => 'name');
	
	# branch according to sortby
	if  ($cgi->param('sortby') eq 'group') { 

		# significant assistance building this routine was provided by:
		#
		#  Jonathan Gorman,
		#  Bruce Van Allen, and
		#  Brad Baxter

		# initialize their profile
		my %profile;
		
		# process of each resource id
		foreach my $resource_id (@resource_ids) {
		
			# get this resource
			my $resource = MyLibrary::Resource->new(id => $resource_id);
			
			# get the url; dumb!
			my @locations = $resource->resource_locations;
			my $location;
			foreach ( @locations ) { $location = $_->location }
			
			# process each related term
			foreach my $term_id ($resource->related_terms) {
			
				# get the term and facet
				my $term = MyLibrary::Term->new(id => $term_id);
				my $facet = MyLibrary::Facet->new(id => $term->facet_id);
				
				# build the profile, finally; seems simple, but...
				$profile{ $facet->facet_name }{ $term->term_name }{ $resource->name } = $location;
				 			
			}
			
		}
		
		# traverse the profile building a set of nested lists
		$resource_list = '<ul>';
		foreach my $facet (sort(keys(%profile))) {

			$resource_list .= "<li><b>$facet</b>";
			my %facets = %{$profile{$facet}};
			$resource_list .= '<ul>';
			foreach my $term (sort(keys(%{$profile{$facet}}))) {

				$resource_list .= "<li>$term";
				my %terms = %{$facets{$term}};
				$resource_list .= '<ol>';
				foreach my $resource (sort(keys(%terms))) {
					
					$resource_list .= "<li><a href='$facets{$term}{$resource}'>$resource</a></li>";
			
				}
				$resource_list .= '</ol></li>';	
		
			}
			$resource_list .= '</ul></li>';
		
		}
		$resource_list .= '</ul>';

	}
	
	elsif (($cgi->param('sortby') eq 'alpha') || (! $cgi->param('sortby'))) {
	
		foreach my $resource_id (@resource_ids) {
		
			# get this resource
			my $resource = MyLibrary::Resource->new(id => $resource_id);
			
			# get the url; dumb!
			my @locations = $resource->resource_locations;
			my $location;
			foreach ( @locations ) { $location = $_->location }
			
			# build a list
			$resource_list .= $cgi->li($cgi->a({-href => $location}, $resource->name) . ' - ' . $resource->note);
	
		}
		
		$resource_list = $cgi->ol($resource_list);

		
	}
	
	# create the output
	$html = &slurp(TEMPLATE);
	$html =~ s/##CONTENT##/&slurp(PATRON)/e;
	$html =~ s/##FIRSTNAME##/$patron->patron_firstname/ge;
	$html =~ s/##LASTNAME##/$patron->patron_surname/ge;
	$html =~ s/##PATRONID##/$patron->patron_id/ge;
	$html =~ s/##RESOURCES##/$resource_list/ge;

}

else {

	# error
	$html  = $cgi->start_html(-title => 'Portal');
	$html .= $cgi->h1('Portal');
	$html .= $cgi->p("Unknown value for cmd ($cmd). Call Eric.");
	$html .= $cgi->end_html;

}


# done
print $cgi->header(-type => 'text/html', -charset => 'utf-8');
$html =~ s/##PATRONS##/&list_patrons/e;
$html =~ s/##VERSION##/MyLibrary->version/e;
print $html;


# subroutines

sub number_of_items { return scalar(MyLibrary::Resource->get_resources(output => 'id')) }


sub slurp {

	# open a file named by the input and return its contents
	my $f = @_[0];
	my $r;
	open (F, "< $f");
	while (<F>) { $r .= $_ }
	close F;
	return $r;

}


sub facet_term_combinations {

	# get all the facets, build a list, and display it
	my @facets = MyLibrary::Facet->get_facets (sort => 'name');
	my $items;
	foreach (@facets) {
	
		# create a list of terms associated with each facet
		my $terms = '<ol>';
		foreach my $id ($_->related_terms(sort => 'name')) {
		
			my $term = MyLibrary::Term->new(id => $id);
			$terms .= '<li><a href="?cmd=' . CMDTERM . '&id=' . $id .  '">' . $term->term_name . '</a> <span style="color: gray"> (' . scalar($term->related_resources) . ' items)</span></li>';
		}
		
		$terms .= '</ol>';
		
		# build the list
		$items .= $cgi->li('<span style="font-weight: bold">' . $_->facet_name, '</span> - ', $_->facet_note, $terms);
		
	}
	
	# done
	return $cgi->ol($items);
	
}



sub list_patrons {

	my $patron_list;
	my @patrons = MyLibrary::Patron->get_patrons();
	foreach (@patrons) { $patron_list .= $cgi->li($cgi->a({-href => './?cmd=patron&id=' . $_->patron_id}, $_->patron_firstname . ' ' . $_->patron_surname)) }
	return $cgi->ol($patron_list);

}

sub remove_duplicates {

  my @list = @_;
  my %seen = ();
  return grep { ! $seen{$_} ++ } @list;

}

